//DUPLEX_RELAY_FIRMWARE should be connected to a known LAN (eg home WiFi)
//has a hostname of 'relay.[MACADD]'
//responds to /open and /close HTTP pages

const uint16_t port = 80; //well known http port
#include <EEPROM.h>
#define EEPROM_SIZE 4096
#define EEPROM_KEY (0xA5)

#if defined(ESP8266)
//for ESP8266:
#include <ESP8266WiFi.h>
#include <ESP8266WiFiMulti.h>
#include <WiFiClient.h>
#include <ESP8266HTTPClient.h>
ESP8266WiFiMulti multi;
WiFiClient client;
#define SOFTRESTART() ESP.restart()
char buf[64]="";
HTTPClient http;
#define HTTPBEGIN(x) http.begin(client,x)
#endif

#if defined(ARDUINO_ARCH_RP2040)
//for Pico:
#include <WiFi.h>
WiFiMulti multi;
#define SOFTRESTART() rp2040.reboot()
#include <HTTPClient.h>
HTTPClient http;
#define HTTPBEGIN(x) http.begin(x)
#endif

//this has the structs/defines etc
#include <DNSServer.h>
//#include <WiFiUdp.h>
#include <ESP8266mDNS.h>        // Include the mDNS library, MDNS object
//WiFiUDP udp;
char ssidName[WL_SSID_MAX_LENGTH]="";
char ssidPass[WL_WPA_KEY_MAX_LENGTH]="";
#define SCAN_LEN (128)
#define BACKSPACE (8)

IPAddress ip,staip; //staip is ours

#define RELAY_MAX (15)
//assume they are all on port 80
char rNames[RELAY_MAX][MDNS_DOMAIN_MAXLENGTH];
IPAddress rIPs[RELAY_MAX];
#define RELAY_EEPROM_START (128)
#define IP_SIZE 4
int addCnt,updateCnt;   //of relay entries
int activeRelay=-1;     //index into RELAY_MAX
#define UPDATE_INTERVAL (60000UL)
unsigned long lastUpdate=0;
String url;
#define LINESIZE (16)

void setup() {
  int c,i,j,k,n,rCount;
  unsigned int m;
  char* s=NULL;
  Serial.begin(115200);
  while(!Serial && millis()<5000){}   //wait for serial with timeout
  Serial.println("Starting");
  EEPROM.begin(EEPROM_SIZE);
  if(EEPROM.read(WL_SSID_MAX_LENGTH+WL_WPA_KEY_MAX_LENGTH)!=EEPROM_KEY){
    saveToEEPROM(); //set initial RAM values
    Serial.println("Setting EEPROM to default values");
  } 
  loadFromEEPROM();
  Serial.println("EEPROM Relay list:");
  printRelayList();
  c=WiFi.scanNetworks();
  if(ssidName[0]){  //avoid connecting to blank
    if(ssidPass[0]){
      multi.addAP(ssidName,ssidPass);
    }else{
      multi.addAP(ssidName,NULL);
    }          
    Serial.printf("Trying to connect to saved network \"%s\".\r\n",ssidName);
  }
  if(multi.run()!=WL_CONNECTED){
    Serial.printf("Cannot connect to saved network \"%s\".\r\n",ssidName);    
    if (!c) {
      Serial.printf("No networks found, rebooting\n");
      delay(2000);
      SOFTRESTART();
    }else{
      Serial.printf("Found %d networks\r\n", c);
      Serial.printf("N: %32s %2s %4s\r\n", "SSID", "CH", "RSSI");
      for (int i = 0; i < c; i++) {
#if defined(ESP8266)
        WiFi.SSID(i).toCharArray(buf,60);
        Serial.printf("%2d %32s %2d %4ld\r\n",i,buf, WiFi.channel(i), WiFi.RSSI(i));
#endif
#if defined(ARDUINO_ARCH_RP2040)
        Serial.printf("%2d %32s %2d %4ld\r\n",i,WiFi.SSID(i), WiFi.channel(i), WiFi.RSSI(i));
#endif
      }
      Serial.println("Enter a number from the network list.");
      while(s==NULL){
        s=scanSerial();
        if(s){
          n=s[0]-'0';
          if(s[1]){n=n+10+s[1]-'0';}
          if((n>=0)&&(n<c)){
            Serial.printf("Enter password for %d:\"%s\".\r\n",n,WiFi.SSID(n));
          }else{
            s=NULL;
            Serial.printf("Network %d not available, enter another number.\r\n",n);
          }
        }
      }
      s=NULL;
      while(s==NULL){
        s=scanSerial();
        if(s){
          for(i=0;i<WL_SSID_MAX_LENGTH;i++){ssidName[i]=WiFi.SSID(n)[i];}
          for(i=0;i<WL_WPA_KEY_MAX_LENGTH;i++){ssidPass[i]=s[i];}
          Serial.println("Connecting...");
          if(ssidPass[0]){
            multi.addAP(ssidName,ssidPass);
          }else{
            multi.addAP(ssidName,NULL);
          }          
          if (multi.run() != WL_CONNECTED) {            
            Serial.printf("Cannot connect to \"%s\"; rebooting.\r\n",WiFi.SSID(n));  
            delay(2000);
            SOFTRESTART();
          }else{
            saveToEEPROM();
            Serial.println("Saved to EEPROM.");
          }
        }
      }
    }
  }
  staip=WiFi.localIP();
  Serial.println("WiFi connected");
  Serial.print("IP address: ");
  Serial.println(staip);    
  Serial.print("DNS:");
  Serial.println(WiFi.gatewayIP());
  Serial.print("Hostname:");
  Serial.println(WiFi.getHostname());  
  if(MDNS.begin(WiFi.getHostname())) {             // Start the mDNS responder
    Serial.print(WiFi.getHostname());
    Serial.println(".local mDNS responder up.");        
  }else{
    Serial.println("mDNS responder did not start.");
  }
  Serial.println("Scanning network.");
  updateHostnames();
  Serial.printf("%d added, %d updated.\r\n",addCnt,updateCnt);
  Serial.println("Updated List:");
  printRelayList();
  if(addCnt|updateCnt){saveToEEPROM();}  
  showHelp();
}

void loop() {
  int i,j,d;
  while(Serial.available()){
    d=Serial.read();
    if(d=='~'){SOFTRESTART();}
    if(d=='?'){showHelp();}
    if((d>='A')&&(d<('A'+RELAY_MAX))){
      activeRelay=d-'A';
      Serial.print("Relay ");
      Serial.write('A'+activeRelay);
      Serial.println(" active.");
    }
    if((d>='a')&&(d<('a'+RELAY_MAX))){
      activeRelay=d-'a';
      Serial.print("Relay ");
      Serial.write('A'+activeRelay);
      Serial.println(" active.");
    }
    if((d=='Z')||(d=='z')){
      Serial.println("Relay List:");
      printRelayList();
    }
    if((d=='Y')||(d=='y')){
      lastUpdate=0;   //force update
      updateHostnames();
      if(addCnt){saveToEEPROM();}  
      Serial.printf("%d added, %d updated.\r\n",addCnt,updateCnt);
      if(addCnt|updateCnt){
        Serial.println("Updated List:");
        printRelayList();
      }
    }
    if((d=='X')||(d=='x')){
      if((activeRelay>=0)&&(activeRelay<RELAY_MAX)){
        rNames[activeRelay][0]=0;
        rIPs[activeRelay]=IPAddress(0,0,0,0);
        Serial.printf("Relay %c deleted.\r\n",'A'+activeRelay);
        saveToEEPROM();
        activeRelay=-1;   //not possible to use
      }else{
        Serial.println("No valid relay selected.");
      }
    }
    if((d=='W')||(d=='w')){
      showEEPROM(); 
    }
    if((d=='0')||(d=='1')){      
      i=doCmd(d-'0');
      if(i==0){
        Serial.println("Retry:");
        if(doCmd(d-'0')==0){
          Serial.println("Request failed");
        }
      }else{
        if(i==-1){    //cases where a retry won't help
          Serial.println("Request failed");
        }
      }
    }
  }
}

int doCmd(char d){
  int state=0;
  int httpcode=0;
  int retval=0;
  if(activeRelay<0){
    Serial.println("No relay selected");
    return -1;
  }
  if(rNames[activeRelay][0]){
    Serial.printf("Requesting relay %c %s to %s.\r\n",'A'+activeRelay,rNames[activeRelay],(d=='1')?"close/energise":"open/de-energise");
    updateHostnames();
    if(rIPs[activeRelay].isSet()){
      url=String("http://")+rIPs[activeRelay].toString()+String("/");
      if(d){
        url=url+String("close");
      }else{
        url=url+String("open");
      }
      state=HTTPBEGIN(url);
      Serial.printf("Requesting %s\r\n",url.c_str());
      if(state){
        httpcode=http.GET();
        Serial.printf("HTTP code %d received.\r\n",httpcode);
        if(httpcode==HTTP_CODE_OK){
          Serial.println("Command done.");
          retval=1;
        }
      }else{
        Serial.println("Cannot connect to server.");
      }
      http.end();
    }else{
      Serial.println("Relay does not have valid IP address.");
      retval=-1;
    }
  }else{
    Serial.println("Relay not in list, cannot set.");
    retval=-1;
  }
  return retval;
}

void updateHostnames(){
  int i,j,k,n;
  unsigned int m;  
  addCnt=0;
  updateCnt=0;
  if((lastUpdate>0)&&((millis()-lastUpdate)<UPDATE_INTERVAL)){    
    //Serial.printf("Last update was only %lu/%lu seconds ago.\r\n",(millis()-lastUpdate)/1000,UPDATE_INTERVAL/1000);
    return;
  }
  lastUpdate=millis();
  //Serial.println("Querying mDNS");
  m=MDNS.queryService("http", "tcp",500); //1000 is default, 500 timeout seems to work
  //Serial.print(m);
  //Serial.println(" mDNS responders found.");
  for(i=0;i<m;i++){
    //Serial.print(m);
    //Serial.print(":  ");
    //Serial.print(MDNS.answerHostname(i));
    //Serial.print("  at  ");
    //Serial.print(MDNS.answerIP(i));
    //Serial.print(":");
    //Serial.println(MDNS.answerPort(i));
    if(MDNS.answerPort(i)==port){
      if(checkHostName(i)){
        for(j=0;j<RELAY_MAX;j++){
          if(strcmp(rNames[j],MDNS.answerHostname(i))==0){  //existing record
            if(rIPs[j]!=MDNS.answerIP(i)){   //IP changed
              updateCnt=updateCnt+1;
              rIPs[j]=MDNS.answerIP(i);   //copy
            }
            j=RELAY_MAX*2;    //bail out on success
            //Serial.println("^Updated");
          }
        }
        if(j<(RELAY_MAX*2)){   //skip if existing/updated
          for(j=0;j<RELAY_MAX;j++){
            if(rNames[j][0]==0){    //find blank
              rIPs[j]=MDNS.answerIP(i);   //copy
              for(k=0;k<MDNS_DOMAIN_MAXLENGTH;k++){
                rNames[j][k]=MDNS.answerHostname(i)[k];
              }
              j=RELAY_MAX;    //bail out on success
              addCnt=addCnt+1;
              //Serial.println("^Added");
            }
          }
        }
      }
    }
  }
  //Serial.printf("Query done, %d added/updated.\r\n",rCount);
  MDNS.removeQuery();
//  Serial.println("Done");
}

void showEEPROM(){
  int i,j;
  char a;
  Serial.println("                       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F");
  for(j=0;j<EEPROM_SIZE;j=j+LINESIZE){  //16 per line
    if(j<16){Serial.write('0');}
    if(j<256){Serial.write('0');}
    if(j<4096){Serial.write('0');}
    Serial.print(j,HEX);
    Serial.write(' ');
    for(i=0;i<LINESIZE;i++){
      a=EEPROM.read(i+j);
      if((a>31)&&(a<127)){  //ascii
        Serial.write(a);
      }else{
        Serial.write('.');
      }
    }
    for(i=0;i<LINESIZE;i++){
      a=EEPROM.read(i+j);
      Serial.write(' ');
      if (a<0x10){Serial.write('0');}          
      Serial.print(a, HEX);
    }
    Serial.println();
  }
}

void showHelp(){
  Serial.println("? for this help");
  Serial.println("~ to reboot");
  Serial.println("Letter A.. to control relay as per list.");
  Serial.println("0 or 1 to set currently active relay.");
  Serial.println("W: show EEPROM");
  Serial.println("X: delete active relay");
  Serial.println("Y: do mDNS scan/update relays");
  Serial.println("Z: show current list of relays");
}

bool checkHostName(unsigned int i){
  if(MDNS.answerHostname(i)[0]!='r'){return 0;}
  if(MDNS.answerHostname(i)[1]!='e'){return 0;}
  if(MDNS.answerHostname(i)[2]!='l'){return 0;}
  if(MDNS.answerHostname(i)[3]!='a'){return 0;}
  if(MDNS.answerHostname(i)[4]!='y'){return 0;}
  if(MDNS.answerHostname(i)[5]!='.'){return 0;}
  //[]6-17 are MAC address, don't care
  if(MDNS.answerHostname(i)[18]!='.'){return 0;}
  if(MDNS.answerHostname(i)[19]!='l'){return 0;}
  if(MDNS.answerHostname(i)[20]!='o'){return 0;}
  if(MDNS.answerHostname(i)[21]!='c'){return 0;}
  if(MDNS.answerHostname(i)[22]!='a'){return 0;}
  if(MDNS.answerHostname(i)[23]!='l'){return 0;}
  if(MDNS.answerHostname(i)[24]!=0){return 0;}
  return 1;
}

void loadFromEEPROM(){
  int i,j;
  int a,d;
  uint8_t o[IP_SIZE]={0,0,0,0};
  for(i=0;i<WL_SSID_MAX_LENGTH;i++){
    ssidName[i]=EEPROM.read(i);
  }
  for(i=0;i<WL_WPA_KEY_MAX_LENGTH;i++){
    ssidPass[i]=EEPROM.read(i+WL_SSID_MAX_LENGTH);
  }  
  for(i=0;i<RELAY_MAX;i++){
    for(j=0;j<MDNS_DOMAIN_MAXLENGTH;j++){
      rNames[i][j]=EEPROM.read(RELAY_EEPROM_START+j+i*MDNS_DOMAIN_MAXLENGTH);
    }    
    for(j=0;j<IP_SIZE;j++){
      a=RELAY_EEPROM_START+RELAY_MAX*MDNS_DOMAIN_MAXLENGTH+i*IP_SIZE+j;
      d=EEPROM.read(a);
      o[j]=d;
    }
    rIPs[i]=IPAddress(o[0],o[1],o[2],o[3]);
  }
}

void saveToEEPROM(){
  int i,j;
  int a,d;
  for(i=0;i<WL_SSID_MAX_LENGTH;i++){
    EEPROM.write(i,ssidName[i]);
  }
  for(i=0;i<WL_WPA_KEY_MAX_LENGTH;i++){
    EEPROM.write(i+WL_SSID_MAX_LENGTH,ssidPass[i]);
  }
  EEPROM.write(WL_SSID_MAX_LENGTH+WL_WPA_KEY_MAX_LENGTH,EEPROM_KEY);  //OK!
  for(i=0;i<RELAY_MAX;i++){
    for(j=0;j<MDNS_DOMAIN_MAXLENGTH;j++){
      EEPROM.write(RELAY_EEPROM_START+j+i*MDNS_DOMAIN_MAXLENGTH,rNames[i][j]);
    }
    for(j=0;j<IP_SIZE;j++){
      a=RELAY_EEPROM_START+RELAY_MAX*MDNS_DOMAIN_MAXLENGTH+i*IP_SIZE+j;
      d=rIPs[i][j];
      EEPROM.write(a,d);
    }
  }
  EEPROM.commit();
  Serial.println("Save to EEPROM done.");
}

char* scanSerial(void){
  static char* s=NULL;						//for input scanning
  static char b[SCAN_LEN]="";
  static int p=0;
	int d;
	if(p==0){b[p]=0;} //if data has been read out, reset
	while(Serial.available()){
		d=Serial.read();
    if((d==3)||(d==27)){  //Ctrl-C/ESC
      b[0]=3;
      b[1]=0;
      p=0;    
      Serial.println("\r\nCancelled\r\n");
      return b;//return empty
    }
    if(d>=' '){
      if(p<SCAN_LEN-2){
      b[p]=d;
      p++;
      b[p]=0; //null term
      Serial.write(d);  //echo
      }
    }else{
      if(d==BACKSPACE){  //backspace
        if(p){
          p--;
          b[p]=0; //delete last
          Serial.write(BACKSPACE);    //back up
          Serial.write(' ');  		 //blank
          Serial.write(BACKSPACE);    //back up again
        }
      }
      if(d==13){Serial.println("");p=0;return b;}
    }
	}
  return 0;
}

void printRelayList(){
  int i;
  for(i=0;i<RELAY_MAX;i++){
    IPAddress result;
    if(rNames[i][0]!=0){
      Serial.printf("%c %s  at  ",i+'A',rNames[i]);
      Serial.println(rIPs[i]);
    }
  }  
}